import json
import asyncio
import logging
from datetime import datetime, timedelta
from fastapi import FastAPI, WebSocket, WebSocketDisconnect, Request, WebSocketException
from fastapi.responses import HTMLResponse, FileResponse
from fastapi.middleware.cors import CORSMiddleware
import fastapi
from starlette.websockets import WebSocketState

app = FastAPI()

# إضافة إعدادات COR''S للسماح بجميع الأصول
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_methods=["*"],
    allow_headers=["*"],
    allow_credentials=True
)

# إعداد سجلات النظام
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger("uvicorn.error")

# قواميس لتخزين بيانات الاتصال وحالة الأجهزة
connected_devices = {}   # device_id -> websocket
devices_status = {}      # device_id -> { "device_id", "serial_number", "mac_address", "status", "last_seen", "pts_info" }

# قفل للتأكّد من أمان الوصول إلى بيانات الأجهزة في بيئة التوازي
lock = asyncio.Lock()

# قيمة العتبة (بالثواني) لتحديد المهلة قبل اعتبار الجهاز غير متصل (يمكن تعديلها حسب الحاجة)
THRESHOLD_SECONDS = 60

# =====================================================
# إضافة endpoint لخدمة الصفحة الرئيسية (index.html)
# =====================================================
@app.get("/", response_class=HTMLResponse)
async def get_index():
    return FileResponse("templates/index.html")

@app.websocket("/ws/{device_id}")
async def websocket_endpoint(websocket: WebSocket, device_id: str):
    try:
        await websocket.accept()
        logger.info(f"New connection accepted from device {device_id}")
        
        # تسجيل الجهاز فوراً عند الاتصال
        async with lock:
            current_time = datetime.utcnow()
            connected_devices[device_id] = websocket
            devices_status[device_id] = {
                "device_id": device_id,
                "serial_number": "غير متوفر",
                "mac_address": "غير متوفر",
                "status": "متصل",
                "last_seen": current_time,
                "pts_info": {
                    "device_type": "بانتظار المعلومات",
                    "firmware_version": "بانتظار المعلومات",
                    "operation_status": "متصل"
                }
            }
            logger.info(f"Device {device_id} added to connected devices. Total devices: {len(connected_devices)}")
            
        # إرسال رسالة الترحيب
        await websocket.send_json({
            "type": "welcome",
            "message": "تم الاتصال بنجاح"
        })

        while True:
            try:
                message = await websocket.receive_text()
                data = json.loads(message)
                logger.debug(f"Received from {device_id}: {data}")
                
                async with lock:
                    if device_id in devices_status:
                        devices_status[device_id]["last_seen"] = datetime.utcnow()
                        
                await handle_device_message(device_id, data)
                
            except WebSocketDisconnect:
                logger.info(f"Device {device_id} disconnected")
                break
            except Exception as e:
                logger.error(f"Error processing message from {device_id}: {str(e)}")
                continue
                
    finally:
        async with lock:
            if device_id in connected_devices:
                connected_devices.pop(device_id)
            if device_id in devices_status:
                devices_status[device_id]["status"] = "غير متصل"
        logger.info(f"Device {device_id} removed. Remaining devices: {len(connected_devices)}")

async def handle_device_message(device_id: str, data: dict):
    """معالجة الرسائل العامة من الجهاز"""
    try:
        if "device_info" in data:
            async with lock:
                if device_id in devices_status:
                    devices_status[device_id].update({
                        "serial_number": data["device_info"].get("serial_number", "غير متوفر"),
                        "mac_address": data["device_info"].get("mac_address", "غير متوفر")
                    })
                    logger.info(f"Updated device info for {device_id}")
        
        if "heartbeat" in data:
            async with lock:
                if device_id in devices_status:
                    devices_status[device_id]["last_seen"] = datetime.utcnow()
                    logger.debug(f"Heartbeat received from {device_id}")
    
    except Exception as e:
        logger.error(f"Error handling device message: {str(e)}")

# إضافة مسار إضافي للتعامل مع الاتصالات من النطاق
@app.websocket("/wss://raito-pts.com")
async def legacy_websocket_endpoint(websocket: WebSocket):
    try:
        logger.info(f"Redirecting connection from {websocket.client.host}")
        # Get device info from headers
        device_id = websocket.headers.get("x-pts-id", "unknown")
        firmware_version = websocket.headers.get("x-pts-firmware-version-datetime", "unknown")
        config_id = websocket.headers.get("x-pts-configuration-identifier", "unknown")
        
        await websocket.accept()
        
        async with lock:
            current_time = datetime.utcnow()
            devices_status[device_id] = {
                "device_id": device_id,
                "serial_number": device_id,  # Use PTS ID as serial number
                "mac_address": config_id,    # Use config ID as MAC address
                "status": "متصل",
                "last_seen": current_time,
                "pts_info": {
                    "device_type": "PTS Device",
                    "firmware_version": firmware_version,
                    "operation_status": "Online"
                }
            }
            connected_devices[device_id] = websocket
            logger.info(f"Legacy connection accepted for device {device_id}")
            
        while True:
            message = await websocket.receive_text()
            try:
                data = json.loads(message)
                logger.info(f"Received data from legacy connection {device_id}: {data}")
                
                # Update device status from PTS messages
                if data.get("Protocol") == "jsonPTS":
                    packets = data.get("Packets", [])
                    for packet in packets:
                        if packet.get("Type") == "UploadStatus":
                            status_data = packet.get("Data", {})
                            async with lock:
                                if device_id in devices_status:
                                    devices_status[device_id]["pts_info"].update({
                                        "firmware_version": status_data.get("FirmwareDateTime", firmware_version),
                                        "operation_status": "Online" if not status_data.get("PowerDownDetected", False) else "Power Down",
                                        "battery": f"{status_data.get('BatteryVoltage', 0)/1000:.2f}V",
                                        "cpu_temp": f"{status_data.get('CpuTemperature', 0)}°C"
                                    })
                                    devices_status[device_id]["last_seen"] = datetime.utcnow()
                
            except json.JSONDecodeError as e:
                logger.error(f"Invalid JSON from legacy connection: {str(e)}")
                continue
                
    except Exception as e:
        logger.error(f"Error in legacy connection: {str(e)}")
    finally:
        if websocket.client_state == WebSocketState.CONNECTED:
            async with lock:
                if device_id in devices_status:
                    devices_status[device_id]["status"] = "غير متصل"
                connected_devices.pop(device_id, None)
            await websocket.close()

# تحديث دالة handle_pts_message لتكون أكثر تفصيلاً
async def handle_pts_message(device_id: str, data: dict):
    try:
        packets = data.get("Packets", [])
        response_data = {}
        
        for packet in packets:
            packet_type = packet.get("Type")
            packet_data = packet.get("Data", {})
            
            if packet.get("Error"):
                logger.error(f"PTS Error: {packet.get('Message')} (Code: {packet.get('Code')})")
                continue
                
            if packet_type == "UploadStatus":
                async with lock:
                    if device_id in devices_status:
                        status_data = packet_data
                        devices_status[device_id]["pts_info"].update({
                            "firmware_version": status_data.get("FirmwareDateTime"),
                            "operation_status": "Offline" if status_data.get("PowerDownDetected") else "Online",
                            "battery": f"{status_data.get('BatteryVoltage', 0)/1000:.2f}V",
                            "cpu_temp": f"{status_data.get('CpuTemperature', 0)}°C",
                            "sd_mounted": status_data.get("SdMounted", False),
                            "startup_time": status_data.get("PtsStartupSeconds", 0),
                            "gps_status": status_data.get("Gps", {}).get("Status", "Unknown"),
                            "datetime": status_data.get("DateTime")
                        })
                        devices_status[device_id]["last_seen"] = datetime.utcnow()
                        logger.debug(f"Updated device status for {device_id}: {devices_status[device_id]}")
            
            elif packet_type == "GetReceiptsResponse":
                response_data['receipts'] = packet_data.get("Receipts", [])
                
            elif packet_type == "GetPricesResponse":
                response_data['prices'] = packet_data.get("Prices", {})
                
            elif packet_type == "GetTankInfoResponse":
                response_data['tanks'] = packet_data.get("Tanks", [])
                
            elif packet_type == "SetPricesResponse":
                response_data['price_update'] = "success"
        
        # إرسال البيانات للعميل إذا كان هناك رد
        if response_data and device_id in connected_devices:
            await connected_devices[device_id].send_json({
                "type": "pts_response",
                "data": response_data
            })
            
    except Exception as e:
        logger.error(f"Error handling PTS message: {str(e)}")

async def send_pts_message(device_id: str, message: dict):
    """
    إرسال رسالة PTS إلى الجهاز عبر WebSocket مع محاولات إعادة المحاولة.
    """
    max_retries = 3
    retry_count = 0
    
    while retry_count < max_retries:
        try:
            async with lock:
                if device_id in connected_devices:
                    ws = connected_devices[device_id]
                    logger.info(f"Sending message to device {device_id}: {message}")
                    await ws.send_json(message)
                    logger.info(f"Successfully sent PTS message to device {device_id}")
                    return True
                else:
                    logger.warning(f"Device {device_id} not connected")
                    return False
        except Exception as e:
            retry_count += 1
            logger.error(f"Attempt {retry_count} failed: {str(e)}")
            if retry_count == max_retries:
                logger.error(f"Failed to send PTS message to device {device_id} after {max_retries} attempts: {e}")
                return False
            await asyncio.sleep(1)

@app.get("/api/devices")
async def get_devices():
    """الحصول على قائمة الأجهزة"""
    try:
        now = datetime.utcnow()
        devices = []
        
        async with lock:
            # تحديث حالة جميع الأجهزة
            for device_id, status in devices_status.items():
                device_data = status.copy()
                
                # تحديث حالة الاتصال
                if device_id in connected_devices:
                    time_diff = (now - status["last_seen"]).total_seconds()
                    device_data["status"] = "متصل" if time_diff <= THRESHOLD_SECONDS else "غير متصل"
                else:
                    device_data["status"] = "غير متصل"
                    
                devices.append(device_data)
        
        logger.info(f"Returning {len(devices)} devices. Connected devices: {len(connected_devices)}")
        return devices
        
    except Exception as e:
        logger.error(f"Error in get_devices: {str(e)}")
        return []

@app.post("/api/disconnect/{device_id}")
async def disconnect_device(device_id: str):
    async with lock:
        if device_id in connected_devices:
            ws = connected_devices.pop(device_id)
            try:
                await ws.close()
            except Exception as e:
                logger.error(f"Error closing websocket for device {device_id}: {e}")
            if device_id in devices_status:
                devices_status[device_id]["status"] = "غير متصل"
                devices_status[device_id]["last_seen"] = datetime.utcnow()
            return {"status": "success", "message": f"تم قطع الاتصال للجهاز {device_id}"}
        else:
            return {"status": "error", "message": "الجهاز غير متصل"}

async def handle_custom_command(device_id: str, command: dict):
    """معالج للأوامر المخصصة"""
    try:
        logger.info(f"Processing custom command for {device_id}: {command}")
        
        if not command.get("command_type"):
            return {"status": "error", "message": "نوع الأمر مطلوب"}

        pts_message = {
            "Protocol": "jsonPTS",
            "Packets": [{
                "Id": 1,
                "Type": command.get("command_type"),
                "Data": command.get("data", {})
            }]
        }
        
        async with lock:
            if device_id not in connected_devices:
                return {"status": "error", "message": "الجهاز غير متصل"}
            
            try:
                ws = connected_devices[device_id]
                await ws.send_json(pts_message)
                logger.info(f"Custom command sent successfully to {device_id}")
                return {
                    "status": "success",
                    "message": f"تم إرسال الأمر {command.get('command_type')} بنجاح"
                }
            except Exception as e:
                logger.error(f"Failed to send custom command: {str(e)}")
                return {"status": "error", "message": str(e)}
    except Exception as e:
        logger.error(f"Error in custom command handler: {str(e)}")
        return {"status": "error", "message": f"خطأ في معالجة الأمر: {str(e)}"}

@app.post("/api/command/{device_id}")
async def send_command(device_id: str, request: Request):
    """معالج الأوامر الرئيسي"""
    try:
        data = await request.json()
        logger.info(f"Command request received for {device_id}: {data}")
        
        async with lock:
            if device_id not in connected_devices:
                return {"status": "error", "message": "الجهاز غير متصل"}
            
            ws = connected_devices[device_id]
            command = data.get("command", "")
            
            try:
                if command == "custom":
                    return await handle_custom_command(device_id, data)
                elif command == "getPTSInfo":
                    message = {
                        "Protocol": "jsonPTS",
                        "Packets": [{
                            "Id": 1,
                            "Type": "GetDeviceInfo",
                            "Data": {}
                        }]
                    }
                    await ws.send_json(message)
                    return {"status": "success", "message": "تم إرسال طلب المعلومات"}
                else:
                    await ws.send_json({"command": command})
                    return {"status": "success", "message": f"تم إرسال الأمر {command}"}
            except Exception as e:
                logger.error(f"Error sending command: {str(e)}")
                return {"status": "error", "message": str(e)}
    except Exception as e:
        logger.error(f"Error in command handler: {str(e)}")
        return {"status": "error", "message": f"خطأ في معالجة الطلب: {str(e)}"}

@app.get("/api/receipts/{device_id}")
async def get_receipts(device_id: str):
    async with lock:
        if device_id in connected_devices:
            try:
                # Get last 24 hours of receipts
                start_date = (datetime.utcnow() - timedelta(days=1)).strftime("%Y-%m-%dT%H:%M:%S")
                end_date = datetime.utcnow().strftime("%Y-%m-%dT%H:%M:%S")
                
                message = {
                    "Protocol": "jsonPTS",
                    "Packets": [{
                        "Id": 1,
                        "Type": "GetReceipts",
                        "Data": {
                            "StartDateTime": start_date,
                            "EndDateTime": end_date
                        }
                    }]
                }
                
                ws = connected_devices[device_id]
                await ws.send_json(message)
                logger.info(f"Sent receipts request for device {device_id}")
                
                return {
                    "status": "success",
                    "message": "تم إرسال طلب الفواتير",
                    "receipts": []  # Initial empty list, will be updated via WebSocket
                }
            except Exception as e:
                logger.error(f"Error requesting receipts from device {device_id}: {e}")
                return {"status": "error", "message": f"خطأ في جلب الفواتير: {str(e)}"}
        return {"status": "error", "message": "الجهاز غير متصل"}

@app.get("/api/tank-info/{device_id}")
async def get_tank_info(device_id: str):
    async with lock:
        if device_id in connected_devices:
            try:
                message = {
                    "Protocol": "jsonPTS",
                    "Packets": [{
                        "Id": 1,
                        "Type": "GetTankInfo",
                        "Data": {}
                    }]
                }
                
                ws = connected_devices[device_id]
                await ws.send_json(message)
                logger.info(f"Sent tank info request for device {device_id}")
                
                return {
                    "status": "success",
                    "message": "تم إرسال طلب معلومات الخزانات",
                    "tanks": []  # Initial empty list, will be updated via WebSocket
                }
            except Exception as e:
                logger.error(f"Error requesting tank info from device {device_id}: {e}")
                return {"status": "error", "message": f"خطأ في جلب معلومات الخزانات: {str(e)}"}
        return {"status": "error", "message": "الجهاز غير متصل"}

@app.get("/api/prices/{device_id}")
async def get_prices(device_id: str):
    async with lock:
        if device_id in connected_devices:
            try:
                message = {
                    "Protocol": "jsonPTS",
                    "Packets": [{
                        "Id": 1,
                        "Type": "GetPrices",
                        "Data": {}
                    }]
                }
                
                ws = connected_devices[device_id]
                await ws.send_json(message)
                logger.info(f"Sent prices request for device {device_id}")
                
                return {
                    "status": "success",
                    "message": "تم إرسال طلب الأسعار",
                    "prices": {"p91": 0, "p95": 0, "diesel": 0}  # Initial values, will be updated via WebSocket
                }
            except Exception as e:
                logger.error(f"Error requesting prices from device {device_id}: {e}")
                return {"status": "error", "message": f"خطأ في جلب الأسعار: {str(e)}"}
        return {"status": "error", "message": "الجهاز غير متصل"}

@app.post("/api/prices/{device_id}")
async def update_prices(device_id: str, request: Request):
    data = await request.json()
    async with lock:
        if device_id in connected_devices:
            ws = connected_devices[device_id]
            try:
                await ws.send_json({
                    "Protocol": "jsonPTS",
                    "Packets": [{
                        "Id": 1,
                        "Type": "SetPrices",
                        "Data": {
                            "91": data["p91"],
                            "95": data["p95"],
                            "diesel": data["diesel"]
                        }
                    }]
                })
                return {"status": "success", "message": "تم تحديث الأسعار بنجاح"}
            except Exception as e:
                logger.error(f"Error updating prices for device {device_id}: {e}")
                return {"status": "error", "message": "خطأ في تحديث الأسعار"}
        return {"status": "error", "message": "الجهاز غير متصل"}

@app.get("/api/pts/{device_id}/status")
async def get_pts_status(device_id: str):
    """الحصول على حالة جهاز PTS"""
    async with lock:
        if device_id in connected_devices:
            try:
                message = {
                    "Protocol": "jsonPTS",
                    "Packets": [{
                        "Id": 1,
                        "Type": "GetStatus",
                        "Data": {}
                    }]
                }
                await send_pts_message(device_id, message)
                return {"status": "success", "message": "تم إرسال طلب الحالة"}
            except Exception as e:
                return {"status": "error", "message": str(e)}
        return {"status": "error", "message": "الجهاز غير متصل"}

@app.post("/api/pts/{device_id}/shift")
async def manage_shift(device_id: str, request: Request):
    """إدارة المناوبات (فتح/إغلاق)"""
    data = await request.json()
    action = data.get("action", "")  # "start" or "end"
    
    if action not in ["start", "end"]:
        return {"status": "error", "message": "العملية غير صالحة"}
        
    async with lock:
        if device_id in connected_devices:
            try:
                message = {
                    "Protocol": "jsonPTS",
                    "Packets": [{
                        "Id": 1,
                        "Type": "ShiftControl",
                        "Data": {
                            "Action": action,
                            "OperatorId": data.get("operator_id", ""),
                            "ShiftNumber": data.get("shift_number", 1)
                        }
                    }]
                }
                await send_pts_message(device_id, message)
                return {"status": "success", "message": f"تم إرسال طلب {action} المناوبة"}
            except Exception as e:
                return {"status": "error", "message": str(e)}
        return {"status": "error", "message": "الجهاز غير متصل"}

@app.post("/api/pts/{device_id}/pump-control")
async def control_pump(device_id: str, request: Request):
    """التحكم في المضخات"""
    data = await request.json()
    pump_number = data.get("pump_number")
    action = data.get("action")  # "enable" or "disable"
    
    if not pump_number or action not in ["enable", "disable"]:  # تصحيح 'أو' إلى 'or'
        return {"status": "error", "message": "معلومات غير صالحة"}
        
    async with lock:
        if device_id in connected_devices:
            try:
                message = {
                    "Protocol": "jsonPTS",
                    "Packets": [{
                        "Id": 1,
                        "Type": "PumpControl",
                        "Data": {
                            "PumpNumber": pump_number,
                            "Action": action
                        }
                    }]
                }
                await send_pts_message(device_id, message)
                return {"status": "success", "message": f"تم إرسال أمر {action} للمضخة {pump_number}"}
            except Exception as e:
                logger.error(f"Error in pump control: {str(e)}")
                return {"status": "error", "message": str(e)}
        return {"status": "error", "message": "الجهاز غير متصل"}

@app.get("/api/pts/{device_id}/daily-report")
async def get_daily_report(device_id: str):
    """الحصول على التقرير اليومي"""
    async with lock:
        if device_id in connected_devices:
            try:
                message = {
                    "Protocol": "jsonPTS",
                    "Packets": [{
                        "Id": 1,
                        "Type": "GetDailyReport",
                        "Data": {
                            "Date": datetime.now().strftime("%Y-%m-%d")
                        }
                    }]
                }
                await send_pts_message(device_id, message)
                return {"status": "success", "message": "تم إرسال طلب التقرير اليومي"}
            except Exception as e:
                return {"status": "error", "message": str(e)}
        return {"status": "error", "message": "الجهاز غير متصل"}

if __name__ == "__main__":
    import uvicorn
    
    uvicorn.run(
        "server:app",
        host="0.0.0.0",
        port=8080,
        reload=True,
        ws_max_size=16777216,
        log_level="debug"  # إضافة تسجيل مفصل للتشخيص
    )