<?php
// Simple WebSocket test server
// Run with: php test-websocket.php
// This simulates price updates for testing the fuel-prices.html page

require __DIR__ . '/../vendor/autoload.php';

use Ratchet\Http\HttpServer;
use Ratchet\Server\IoServer;
use Ratchet\WebSocket\WsServer;
use Ratchet\MessageComponentInterface;
use Ratchet\ConnectionInterface;

class FuelPriceSimulator implements MessageComponentInterface
{
    protected $clients;
    protected $prices = [
        "P91" => 27.50,
        "P95" => 23.99,
        "Diesel" => 23.99
    ];

    public function __construct()
    {
        $this->clients = new \SplObjectStorage;
        echo "FuelPriceSimulator started.\n";
    }

    public function onOpen(ConnectionInterface $conn)
    {
        $this->clients->attach($conn);
        echo "New connection! ({$conn->resourceId})\n";
        
        // Send connection confirmation
        $conn->send(json_encode([
            'type' => 'connection',
            'status' => 'connected',
            'client_id' => $conn->resourceId
        ]));
        
        // Send initial price data
        $this->sendPriceUpdate($conn);
        
        // Start a timer to randomly update prices
        $this->startPriceUpdateSimulation($conn);
    }

    public function onMessage(ConnectionInterface $from, $msg)
    {
        echo "Received message from {$from->resourceId}: {$msg}\n";
        // Just echo back the message for testing
        $from->send($msg);
    }

    public function onClose(ConnectionInterface $conn)
    {
        $this->clients->detach($conn);
        echo "Connection {$conn->resourceId} has disconnected\n";
    }

    public function onError(ConnectionInterface $conn, \Exception $e)
    {
        echo "An error has occurred: {$e->getMessage()}\n";
        $conn->close();
    }
    
    protected function sendPriceUpdate(ConnectionInterface $conn)
    {
        // Create a PTS protocol message with fuel prices
        $data = [
            "Protocol" => "jsonPTS",
            "Packets" => [[
                "Id" => 1,
                "Type" => "SetFuelGradesConfiguration",
                "Data" => [
                    "FuelGrades" => [[
                        "Id" => 1,
                        "Name" => "P91",
                        "Price" => $this->prices["P91"]
                    ], [
                        "Id" => 2,
                        "Name" => "P95",
                        "Price" => $this->prices["P95"]
                    ], [
                        "Id" => 3,
                        "Name" => "Diesel",
                        "Price" => $this->prices["Diesel"]
                    ]]
                ]
            ]]
        ];
        
        $conn->send(json_encode($data));
        echo "Sent price update: P91={$this->prices["P91"]}, P95={$this->prices["P95"]}, Diesel={$this->prices["Diesel"]}\n";
    }
    
    protected function startPriceUpdateSimulation(ConnectionInterface $conn)
    {
        // Set up a timer to randomly update prices every 5-10 seconds
        $loop = \React\EventLoop\Loop::get();
        $loop->addPeriodicTimer(rand(5, 10), function() use ($conn) {
            // Randomly select a fuel type to update
            $fuelTypes = array_keys($this->prices);
            $randomFuel = $fuelTypes[array_rand($fuelTypes)];
            
            // Generate a new random price (with small variation)
            $currentPrice = $this->prices[$randomFuel];
            $newPrice = round($currentPrice + (rand(-100, 100) / 100), 2);
            $newPrice = max(15.00, $newPrice); // Ensure price doesn't go below 15
            
            // Update price
            $this->prices[$randomFuel] = $newPrice;
            
            // Send update to client
            $this->sendPriceUpdate($conn);
        });
    }
}

echo "Starting WebSocket server on port 8080...\n";

$server = IoServer::factory(
    new HttpServer(
        new WsServer(
            new FuelPriceSimulator()
        )
    ),
    8080
);

echo "WebSocket server started successfully!\n";
echo "Listening on port 8080...\n";

$server->run(); 